const models = require('../models/afiliados')
const bcrypt = require('bcryptjs');
const yup = require('yup')
const logger = require('../services/loggerService'); 
const response = require("../constants/response");
const sendRequestOnMicroservices = require("../helpers/sendRequestOnMicroservices"); 
const StrengthSchecker = require('../helpers/StrengthSchecker');

module.exports.getAfiliados = async function(req, res, next) {
  try {  

      logger("SERVIDOR:").info(`Buscar os afiliados`)
      const {pagina, limite, total_registros, nome_afiliado = '', email_afiliado = '', user_name = '', entidade = ''} = req.query
      const results = await models.getAfiliados(pagina, limite, total_registros, nome_afiliado, email_afiliado, user_name, entidade)
      
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliados").error(`Erro buscar afiliados ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getAfiliadosId = async function(req, res, next) {
  try {  

      logger("SERVIDOR:getAfiliadosId").info(`Buscar os afiliados pelo ID`)
      const {id_afiliado} = req.params
      const results = await models.getAfiliadosId(id_afiliado)
      
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliadosId").error(`Erro buscar afiliados os afiliados pelo ID ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getAfiliadosIdIdentificador = async function(req, res, next) {
  try {  

      logger("SERVIDOR:getAfiliadosIdIdentificador").info(`Buscar os afiliados pela entidade e sua identificação`)
      const {id_identificacao_afiliados} = req.params
      const results = await models.getAfiliadosIdIdentificador(id_identificacao_afiliados)
      
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliadosIdIdentificador").error(`Erro buscar afiliados pela entidade e sua identificação ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getAfiliadosIdComEntidade = async function(req, res, next) {
  try {  

      logger("SERVIDOR:getAfiliadosIdComEntidade").info(`Buscar os afiliados pela entidade`)
      const {id_afiliado, entidade} = req.params
      const results = await models.getAfiliadosIdComEntidade(id_afiliado, entidade)
      
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliadosIdComEntidade").error(`Erro buscar afiliados pela entidade ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getAfiliadosHash = async function(req, res, next) {
  try{
    logger("SERVIDOR:getAfiliadosHash").info(`Buscar dados do hash do cliente`)
    const {hash} = req.params
    const results = await models.getAfiliadosHash(hash);
    res.status(results.statusCode).json(results)
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:getAfiliadosHash").error(`Erro ao buscar clientes ${error.message}`)
  }
    
}

module.exports.getAfiliadosComEntidadeIdentificadores = async function(req, res, next) {
  try {  

      logger("SERVIDOR:getAfiliadosComEntidadeIdentificadores").info(`Buscar os afiliados pela entidade e sua identificação`)
      const {pagina, limite} = req.query
      const {entidade} = req.params
      const results = await models.getAfiliadosComEntidadeIdentificadores(entidade, pagina, limite)
      
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliadosComEntidadeIdentificadores").error(`Erro buscar afiliados pela entidade e sua identificação ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getAfiliadosIdComEntidadeIdentificador = async function(req, res, next) {
  try {  

      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificador").info(`Buscar os afiliados pela entidade e sua identificação`)
      const {id_identificacao_afiliados, entidade} = req.params
      const results = await models.getAfiliadosIdComEntidadeIdentificador(id_identificacao_afiliados, entidade)
      
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificador").error(`Erro buscar afiliados pela entidade e sua identificação ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getAfiliadosIdComEntidadePermissoes = async function(req, res, next) {
  try {  

      logger("SERVIDOR:getAfiliadosIdComEntidadePermissoes").info(`Buscar os afiliados pela entidade e suas permissões`)
      const {id_afiliado, entidade} = req.params
      const results = await models.getAfiliadosIdComEntidadePermissoes(id_afiliado, entidade)
      
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliadosIdComEntidadePermissoes").error(`Erro buscar afiliados pela entidade e suas permissões ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getAfiliadosIdComEntidadeIdentificadorAfiliados = async function(req, res, next) {
  try {  

      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificadorAfiliados").info(`Buscar os afiliados pela entidade e sua identificação`)
      const {pagina, limite, nome_afiliado = '', email_afiliado = '', user_name = ''} = req.query
      const {id_identificacao_afiliados, entidade} = req.params
      const results = await models.getAfiliadosIdComEntidadeIdentificadorAfiliados(id_identificacao_afiliados, entidade, pagina, limite, nome_afiliado, email_afiliado, user_name)
      
      res.status(results.statusCode).json(results)

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificadorAfiliados").error(`Erro buscar afiliados pela entidade e sua identificação ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.getAfiliadosEntidade = async function(req, res, next) {
  try {  

      logger("SERVIDOR:getAfiliadosEntidade").info(`Buscar os afiliados pela entidade`)
      const {entidade} = req.params
      const {pagina, limite, nome_afiliado = '', email_afiliado = '', user_name = ''} = req.query
      const results = await models.getAfiliadosEntidade(entidade, pagina, limite, limite, nome_afiliado, email_afiliado, user_name)

      res.status(results.statusCode).json(results)  

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:getAfiliadosEntidade").error(`Erro buscar afiliados pela entidade ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}

module.exports.postAfiliados = async function(req, res, next) {
  try {

      logger("SERVIDOR:postAfiliados").info(`Iniciando cadastrado o afiliado`)
    
      const dados =  req.body

      const schemaAfiliados = yup.object().shape({
        email_afiliado: yup.string().email().required(),
        user_pass: yup.string().required(),
        user_pass_confirmar_senha: yup.string().oneOf([yup.ref("user_pass")]).required(),
        user_name: yup.string().min(5).required(),
        nome_afiliado: yup.string().min(10).required(),
        grupo_identificacao: yup.number().required(),
        cliente_id: yup.number().required(),
      }) 

      logger("SERVIDOR:postAfiliados").debug(`Á validar os dados ${JSON.stringify(dados)}`)
      const validar = await schemaAfiliados.validate(dados)

      logger("SERVIDOR:postAfiliados").debug(`Fortificando a senha`)
      const passCheck = await StrengthSchecker(validar.user_pass)

      if(passCheck.bg === "error"){

        logger("SERVIDOR:postAfiliados").info(`Senha para o cliente é muito fraca`)         
        const rs = response("erro", 406, "Senha para o cliente é muito fraca");
        res.status(rs.statusCode).json(rs)         

        return
      }

      var salt = bcrypt.genSaltSync(10);
      var hash = bcrypt.hashSync(validar.user_pass, salt);
      
      delete validar.user_pass_confirmar_senha
    
      const result = await models.postAfiliados({...validar, user_pass:hash}, req)

      var lg = result.logs
      delete result.logs
      
      res.status(result.statusCode).json(result)
      
      if(result.status == "sucesso"){
        sendRequestOnMicroservices({lg})
      }   

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:postClientes").error(`Erro ao cadastrar o cliente ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
  }
    
}

module.exports.postAfiliadosIdentificador = async function(req, res, next) {
  try {

      logger("SERVIDOR:postAfiliadosIdentificador").info(`Iniciando cadastrado o identifador de afiliados`)
      
      const dados =  req.body

      const schemaIdentificadorAfiliados = yup.object().shape({
        nome_identificacao: yup.string().min(5).required(),
        entidade_das_afiliadas: yup.string().min(5).required(),
      })

      logger("SERVIDOR:postAfiliadosIdentificador").debug(`Á validar os dados ${JSON.stringify(dados)}`)
      const validar = await schemaIdentificadorAfiliados.validate(dados)
      
      const result = await models.postAfiliadosIdentificador(validar, req)
      
      var lg = result.logs
      delete result.logs
      
      res.status(result.statusCode).json(result)
      
      if(result.status == "sucesso"){
        sendRequestOnMicroservices({lg})
      }  

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:postClientes").error(`Erro ao cadastrar o cliente ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
  }
    
}

module.exports.postAfiliadosLogin = async function(req, res, next) {
  try {

      logger("SERVIDOR:postAfiliadosLogin").info(`Iniciando cadastrado o identifador de afiliados`)
        
      const dados =  req.body

      const schemaLoginAfiliados = yup.object().shape({
        user_name: yup.string().required(),
        user_pass: yup.string().required(),
      })      

      logger("SERVIDOR:postAfiliadosLogin").debug(`Á validar os dados ${JSON.stringify(dados)}`)
      const validar = await schemaLoginAfiliados.validate(dados)

      const result = await models.postAfiliadosLogin(validar, req) 

      var lg = result.logs
      delete result.logs
      
      res.status(result.statusCode).json(result)
      
      if(result.status == "sucesso"){
        sendRequestOnMicroservices({lg})
      } 

      
  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:postAfiliadosLogin").error(`Erro cadastrado o identifador de afiliados ${error.message}`)
      
      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
  }
    
}

module.exports.postAfiliadosLogout = async function(req, res, next) {
  try {

      logger("SERVIDOR:postAfiliadosLogout").info(`Iniciando realizar logout afiliados`)
          
      const dados =  req.body

      const schemaLoginAfiliados = yup.object().shape({
        id_afiliado: yup.number().required(),
      })      

      logger("SERVIDOR:postAfiliadosLogout").debug(`Á validar os dados ${JSON.stringify(dados)}`)
      const validar = await schemaLoginAfiliados.validate(dados)
      const result = await models.postAfiliadosLogout(validar.id_afiliado, req)

      var lg = result.logs
      delete result.logs
      
      res.status(result.statusCode).json(result)
      
      if(result.status == "sucesso"){
        sendRequestOnMicroservices({lg})
      }

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:postAfiliadosLogout").error(`Erro realizar logout afiliados ${error.message}`)
      
      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
  }
    
}

module.exports.patchAfiliadosPermissoes = async function(req, res, next) {
    try {  

        logger("SERVIDOR:patchAfiliadosPermissoes").info(`Iniciando permissões do afiliados`)
        const {id_afiliado} = req.params
        const dados = req.body

        const schemaPermissoesAfiliados = yup.object().shape({
          pagina_referencia: yup.mixed().oneOf(['true', 'false']),
          pagina_pagamento: yup.mixed().oneOf(['true', 'false']),
          pagina_contabilistico: yup.mixed().oneOf(['true', 'false']),
          pagina_relatorio: yup.mixed().oneOf(['true', 'false']),
          pagina_servico_api: yup.mixed().oneOf(['true', 'false']),
          pagina_faq: yup.mixed().oneOf(['true', 'false']),
          pagina_afiliado: yup.mixed().oneOf(['true', 'false']),
          pagina_sms: yup.mixed().oneOf(['true', 'false']),
          pagina_token: yup.mixed().oneOf(['true', 'false']),
          pagina_webwook:  yup.mixed().oneOf(['true', 'false']),
          pagina_notificacao: yup.mixed().oneOf(['true', 'false']),
          pagina_perfil: yup.mixed().oneOf(['true', 'false']),
          pagina_pagamentos_online: yup.mixed().oneOf(['true', 'false']),
          pagina_relatorios_online: yup.mixed().oneOf(['true', 'false']),
          pagina_sair: yup.mixed().oneOf(['true', 'false'])
         })      
  
        logger("SERVIDOR:patchAfiliadosPermissoes").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaPermissoesAfiliados.validate(dados)
        
        const result = await models.patchAfiliadosPermissoes(id_afiliado, validar, req)

        var lg = result.logs
        delete result.logs
        
        res.status(result.statusCode).json(result)
        
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg})
        }
      
    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchAfiliadosPermissoes").error(`Erro ao actualizar permissões do afiliados ${error.message}`)
        
        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
    }
    
}
  
module.exports.patchAfiliados = async function(req, res, next) {
    try {  

        logger("SERVIDOR:patchAfiliados").info(`Iniciando actualização do afiliado`)
      
        const {id_afiliado} = req.params
        const dados =  req.body

        const schemaAfiliados = yup.object().shape({
          email_afiliado: yup.string().email().required(),
          user_pass: yup.string(),
          user_name: yup.string().min(5).required(),
          nome_afiliado: yup.string().min(10).required(),
          grupo_identificacao: yup.number().required(),
          activacao: yup.mixed().oneOf(['true', 'false']).required(),
        })

        logger("SERVIDOR:patchAfiliados").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaAfiliados.validate(dados)

        if(Object.keys(validar).includes('user_pass')){          
        
          logger("SERVIDOR:patchClientes").debug(`Fortificando a senha`)
          const passCheck = await StrengthSchecker(validar?.user_pass)
          
          if(passCheck.bg === "error"){

            logger("SERVIDOR:patchClientes").info(`Senha para o cliente é muito fraca`)         
            const rs = response("erro", 406, "Senha para o cliente é muito fraca");
            res.status(rs.statusCode).json(rs)         
  
            return
          }

          var salt = bcrypt.genSaltSync(10);
          var hash = bcrypt.hashSync(validar.user_pass, salt);
          validar.user_pass = hash

        }
        
        const result = await models.patchAfiliados(id_afiliado, validar, req)

        var lg = result.logs
        delete result.logs
        
        res.status(result.statusCode).json(result)
        
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg})
        }

    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchAfiliados").error(`Erro ao actualização afiliados ${error.message}`)
        
        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
    }
    
}
  
module.exports.patchAfiliadosAlternarEstado = async function(req, res, next) {
    try {  

        logger("SERVIDOR:patchAfiliadosAlternarEstado").info(`Iniciando actualização do afiliado`)
      
        const {id_afiliado} = req.params
        const dados =  req.body

        const schemaAfiliados = yup.object().shape({
          activacao: yup.mixed().oneOf(['true', 'false']).required(),
        })

        logger("SERVIDOR:patchAfiliadosAlternarEstado").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaAfiliados.validate(dados)
        
        const result = await models.patchAfiliados(id_afiliado, validar, req)

        var lg = result.logs
        delete result.logs
        
        res.status(result.statusCode).json(result)
        
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg})
        }

    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchAfiliadosAlternarEstado").error(`Erro ao actualização afiliados ${error.message}`)
        
        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
    }
    
}

module.exports.patchAfiliadosMudarSenha = async function(req, res, next) {
    try {  

        logger("SERVIDOR:patchAfiliadosMudarSenha").info(`Iniciando actualização da senha do afiliado`)
      
        const {id_afiliado} = req.params
        const dados =  req.body

        const schemaAfiliados = yup.object().shape({
          user_pass: yup.string().required(),
          user_pass_confirmar_senha: yup.string().oneOf([yup.ref("user_pass")]).required(),
        })

        logger("SERVIDOR:patchAfiliadosMudarSenha").debug(`Á validar os dados ${JSON.stringify(dados)}`)
        const validar = await schemaAfiliados.validate(dados)

        logger("SERVIDOR:patchAfiliadosMudarSenha").debug(`Fortificando a senha`)
        const passCheck = await StrengthSchecker(validar.user_pass)

        if(passCheck.bg === "error"){

          logger("SERVIDOR:patchAfiliadosMudarSenha").info(`Senha para o cliente é muito fraca`)         
          const rs = response("erro", 406, "Senha para o cliente é muito fraca");
          res.status(rs.statusCode).json(rs)         

          return
        }
        
        var salt = bcrypt.genSaltSync(10);
        var hash = bcrypt.hashSync(validar.user_pass, salt);
        dados.user_pass = hash
        
        delete validar.user_pass_confirmar_senha
        
        const result = await models.patchAfiliados(id_afiliado, dados, req)

        var lg = result.logs
        delete result.logs
        
        res.status(result.statusCode).json(result)
        
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg})
        }

    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:patchAfiliadosMudarSenha").error(`Erro ao actualizar a senha do afiliados ${error.message}`)
        
        if(error?.path){
          const rs = response("erro", 412, error.message);
          res.status(rs.statusCode).json(rs)        
        }else{  
          const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
          res.status(rs.statusCode).json(rs)
        }
    }
    
}
  
module.exports.deleteAfiliados = async function(req, res, next) {
    try {  

        logger("SERVIDOR:deleteAfiliados").info(`Inicio da exlusão do afiliado`)
        const {id_afiliado} = req.params

        const result = await models.deleteAfiliados(id_afiliado, req)

        var lg = result.logs
        delete result.logs
        
        res.status(result.statusCode).json(result)
        
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg})
        }

    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:deleteAfiliados").error(`Erro ao deleter afiliados ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}