const models = require('../models/referencias')
const yup = require('yup')
const logger = require('../services/loggerService'); 
const response = require("../constants/response");
const path = require('path');
const fs = require('fs');
const { parse, isDate } = require("date-fns");
const moment = require("moment");
const sendRequestOnMicroservices = require("../helpers/sendRequestOnMicroservices"); 


function parseDateString(value, originalValue) {
  const parsedDate = isDate(new Date(originalValue))
    ? new Date(originalValue)
    : parse(originalValue, "yyyy-MM-dd", new Date());

  return parsedDate;
}

module.exports.getReferencias = async function(req, res, next) {
    
    try {
        logger("SERVIDOR:Clientes").info("Buscar clientes")
        const {pagina, limite, total_registros, limite_pagamento = '', entidade = '', referencia = '', estado = '', usabilidade = '', fixo = '', opcional_1 = '', opcional_2 = '', opcional_3 = '', nib = ''} = req.query
        const results = await models.getReferencias(pagina, limite, total_registros, limite_pagamento, entidade, referencia, estado, usabilidade, fixo, opcional_1, opcional_2, opcional_3, nib, req)
        res.status(results.statusCode).json(results)
          
    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:Clientes").error(`Erro ao buscar clientes ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}

module.exports.getReferenciaId = async function(req, res, next) {
    
    try {
        const {id_referencia} = req.params
        const results = await models.getReferenciaId(id_referencia, req)
        res.status(results.statusCode).json(results)
          
    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:Clientes").error(`Erro ao buscar clientes ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}

module.exports.getReferenciaReferencia = async function(req, res, next) {
    
    try {
        const {num_referencia} = req.params
        const results = await models.getReferenciaReferencia(num_referencia, req)
        res.status(results.statusCode).json(results)
          
    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:Clientes").error(`Erro ao buscar clientes ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}

module.exports.getReferenciasEntidade = async function(req, res, next) {
    
    try {
      const {entidade_cliente} = req.params
      const {pagina, limite, total_registros, limite_pagamento = '', referencia = '', estado = '', usabilidade = '', fixo = '', opcional_1 = '', opcional_2 = '', opcional_3 = '', nib = '', criacao = ''} = req.query
      const results = await models.getReferenciasEntidade(entidade_cliente, pagina,  limite, total_registros, limite_pagamento, referencia, estado, usabilidade, fixo, opcional_1, opcional_2, opcional_3, nib, criacao, req)
      res.status(results.statusCode).json(results)
          
    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:Clientes").error(`Erro ao buscar clientes ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}

module.exports.getReferenciasEntidadeGrupoAfiliados = async function(req, res, next) {
    
    try {
      const {entidade_cliente, grupo_identificacao} = req.params
      const {pagina, limite, total_registros} = req.query
      const results = await models.getReferenciasEntidadeGrupoAfiliados(entidade_cliente, grupo_identificacao, pagina,  limite, total_registros, req)
      res.status(results.statusCode).json(results)
          
    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:Clientes").error(`Erro ao buscar clientes ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}

exports.importarXML = async function (req, res){
  try {
      
          
        let file_xml = '';
        const dados = req.body
        const headers = req.headers

        const schemaImportRef = yup.object().shape({
          id_clientes: yup.number().required()
        })

        const schemaImportRefHeaders = yup.object().shape({
          entidade: yup.string().min(5).required()
        })
        
        const validar = await schemaImportRef.validate(dados)
        const validarHeaders = await schemaImportRefHeaders.validate(headers)
        
        if(!validarHeaders.entidade) {
            const rs = response("erro", 406, "Entidade desconhecia");
            res.status(rs.statusCode).json(rs)
            return  rs
        }

        if (req.files) {
              
              let targetFile = req.files.file_xml
              const ext =  path.extname(targetFile.name)
              const filename = targetFile.md5+ext
              file_xml =  filename;


              if(ext !== ".xml"){
                  const rs = response("erro", 406, "Ficheito errado");
                  res.status(rs.statusCode).json(rs)
                  return  rs
              }

              if(!fs.existsSync(path.join(__dirname, `../assets/excel/${id_clientes}`))){
                fs.mkdirSync(path.join(__dirname, `../assets/excel/${id_clientes}`))
              }
              
              targetFile.mv(path.join(__dirname, `../assets/excel/${id_clientes}`,filename), async (err) => {
                  
                  if (err) {
                      const rs = response("erro", 400, "Algo aconteceu de errado. Tente outra vez!");
                      res.status(rs.statusCode).json(rs)
                      return  rs
                  }

                  const data = fs.readFileSync(path.join(__dirname, `../assets/excel/${id_clientes}`,filename), 'utf8')

                  const result = await models.importarXML(validarHeaders.entidade, validar.id_clientes, data, req)

                  var wk = result.webhook
                  var lg = result.logs
                  var nt = result.notification
                  var ft = result.mft
                  
                  delete result.webhook
                  delete result.logs
                  delete result.notification
                  delete result.mft
                  
                  
                  res.status(result.statusCode).json(result)
                  if(result.status == "sucesso"){
                      sendRequestOnMicroservices({lg, nt, wk, ft})
                  }
                        
            });
            
        }else{
          const rs = response("erro", 400, "Algo aconteceu de errado. Tente outra vez! Selecionou sua imagem?");
          res.status(rs.statusCode).json(rs)
          return rs
        }
              
      
  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:postClientes").error(`Erro ao cadastrar o cliente ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
  }
}

module.exports.getReferenciaReferenciaEntidade = async function(req, res, next) {
    
    try {
        const {num_referencia, numero_entidade} = req.params
        const results = await models.getReferenciaReferenciaEntidade(num_referencia, numero_entidade, req)
        res.status(results.statusCode).json(results)
          
    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:Clientes").error(`Erro ao buscar clientes ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}

module.exports.getReferenciasDias = async function(req, res, next) {
    
    try {
        const {dia_inicio, dia_final} = req.params
        const {pagina, limite, total_registros, limite_pagamento = '', entidade = '', referencia = '', estado = '', usabilidade = '', fixo = '', opcional_1 = '', opcional_2 = '', opcional_3 = '', nib = ''} = req.query
        const results = await models.getReferenciasDias(dia_inicio, dia_final, pagina, limite, total_registros, limite_pagamento, entidade, referencia, estado, usabilidade, fixo, opcional_1, opcional_2, opcional_3, nib, req)
        res.status(results.statusCode).json(results)
          
    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:Clientes").error(`Erro ao buscar clientes ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}

module.exports.getReferenciasEntidadeDias = async function(req, res, next) {
    
    try {
        const {entidade_cliente, dia_inicio, dia_final} = req.params
        const {pagina, limite, total_registros, limite_pagamento = '', referencia = '', estado = '', usabilidade = '', fixo = '', opcional_1 = '', opcional_2 = '', opcional_3 = '', nib = ''} = req.query
        const results = await models.getReferenciasEntidadeDias(entidade_cliente, dia_inicio, dia_final, pagina, limite, total_registros, limite_pagamento, referencia, estado, usabilidade, fixo, opcional_1, opcional_2, opcional_3, nib, req)
        res.status(results.statusCode).json(results)
          
    } catch (error) {
        console.error(error.message)
        logger("SERVIDOR:Clientes").error(`Erro ao buscar clientes ${error.message}`)
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
    }
}

module.exports.postReferenciaImport = async function(req) {
  try {
    
    let dados =  req.body

    let c1 = yup.string()
    if(dados?.opcional_1){
      yup.string().matches(/[a-zA-Z]{2,}=[\w{2,}\.\-\,\d\s]{1,}/ig, "Deve colocar correctamente dos dados (CAMPO 1). EX: campo=valor")
    }

    let c2 = yup.string()
    if(dados?.opcional_2){
      yup.string().matches(/[a-zA-Z]{2,}=[\w{2,}\.\-\,\d\s]{1,}/ig, "Deve colocar correctamente dos dados (CAMPO 2). EX: campo=valor")
    }

    let c3 = yup.string()
    if(dados?.opcional_3){
      yup.string().matches(/[a-zA-Z]{2,}=[\w{2,}\.\-\,\d\s]{1,}/ig, "Deve colocar correctamente dos dados (CAMPO 3). EX: campo=valor")
    }

    for (const prop in dados ) {    
      dados[prop] = dados[prop].toString()
    }
    
    if(!dados.hora_limite_pagamento) dados.hora_limite_pagamento =  "23:59"

    if(dados.montante_limitado_para_pagamento_longo && dados.montante_fixo) res.status(422).json({status:"erro", mensagem:"Montante fixo e o Montante limitado não podem ser parametrizados juntos."})    
    
    const refSchemaReg1 = yup.object().shape({
      entidade_cliente: yup.string().min(5).required(),
      tipo_de_registro: yup.number().min(1).max(3).required(),
      montante_fixo: yup.number().min(0).max(19999999),
      montante_limitado_para_pagamento_longo: yup.number().min(0),
      codigo_de_processamento: yup.string().required(),
      num_referencia: yup.string().min(1).max(15),
      indicador_de_produtos: yup.string().required(),
      data_limite_pagamento: yup.date().transform(parseDateString).required(),
      hora_limite_pagamento: yup.string().test(
        "date validation",
        "Invalid date or format. The string should be a valid HH:mm format.",
        val => {
          if (!val) {
            return true;
          }
          return moment(val, "HH:mm", true).isValid();
        }
      ),
      indicador_produto_id: yup.string().required(),
      gerado_por_afiliado: yup.string(),
      opcional_1: c1,
      opcional_2: c2,
      opcional_3: c3 
    })
 
        
    const refSchemaReg3 = yup.object().shape({
      entidade_cliente: yup.string().min(5).required(),
      tipo_de_registro:  yup.number().min(1).max(3).required(),
      codigo_de_processamento: yup.string().required(),
      num_referencia: yup.string().min(1).max(9),
      indicador_de_produtos: yup.string().required(),
      data_limite_pagamento: yup.date().transform(parseDateString).required(),
      data_inicio_de_pagamento: yup.date().transform(parseDateString).required(),
      montante_minimo: yup.number().min(100).required(),
      montante_maximo: yup.number().min(100).max(19999999).required(), 
      codigo_de_cliente: yup.string().required(),
      numero_de_linhas : yup.number().required().max(4),
      textos_para_talao: yup.string().required().max(40),
      indicador_produto_id: yup.string().required(),
      gerado_por_afiliado: yup.string(),
      opcional_1: c1,
      opcional_2: c2,
      opcional_3: c3     
    })
    
    
    if(dados.tipo_de_registro == "1"){
      dados = {
            entidade_cliente: dados.entidade_cliente,
            tipo_de_registro: dados.tipo_de_registro,
            montante_fixo: dados.montante_fixo || 0,
            montante_limitado_para_pagamento_longo: dados.montante_limitado_para_pagamento_longo || 0,
            codigo_de_processamento: dados.codigo_de_processamento,
            num_referencia: dados.num_referencia,
            indicador_de_produtos: dados.indicador_de_produtos,
            data_limite_pagamento: dados.data_limite_pagamento,
            hora_limite_pagamento: dados.hora_limite_pagamento,
            indicador_produto_id: dados.indicador_produto_id,
            gerado_por_afiliado: dados.gerado_por_afiliado || 0,
            opcional_1: dados.opcional_1,
            opcional_2: dados.opcional_2,
            opcional_3: dados.opcional_3,
        }
        
        const validar = await refSchemaReg1.validate(dados)
        const result = await models.postReferencia({...validar, data_limite_pagamento: new Date(validar.data_limite_pagamento).toISOString().split('T')[0]}, req)  
         
        
        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        var ft = result.mft
        
        delete result.webhook
        delete result.logs
        delete result.notification
        delete result.mft
        
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){
            sendRequestOnMicroservices({lg, nt, wk, ft})
        }
                
    }
    
    if(dados.tipo_de_registro == "3"){
      dados = {
            entidade_cliente: dados.entidade_cliente,
            tipo_de_registro: dados.tipo_de_registro,
            codigo_de_processamento: dados.codigo_de_processamento,
            num_referencia: dados.num_referencia,
            indicador_de_produtos: dados.indicador_de_produtos,
            data_limite_pagamento: dados.data_limite_pagamento,
            data_inicio_de_pagamento: dados.data_inicio_de_pagamento,
            montante_minimo: dados.montante_minimo,
            montante_maximo: dados.montante_maximo,
            codigo_de_cliente: dados.codigo_de_cliente,
            numero_de_linhas : dados.numero_de_linhas,
            textos_para_talao: dados.textos_para_talao,
            indicador_produto_id: dados.indicador_produto_id,
            gerado_por_afiliado: dados.gerado_por_afiliado || 0,
            opcional_1: dados.opcional_1,
            opcional_2: dados.opcional_2,
            opcional_3: dados.opcional_3,
            
        }

        
        const validar = await refSchemaReg3.validate(dados)
        const result = await models.postReferencia({...validar, data_inicio_de_pagamento: new Date(validar.data_inicio_de_pagamento).toISOString().split('T')[0], data_limite_pagamento: new Date(validar.data_limite_pagamento).toISOString().split('T')[0]}, req)     
        
        
        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        var ft = result.mft
        
        delete result.webhook
        delete result.logs
        delete result.notification
        delete result.mft
        
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){
            sendRequestOnMicroservices({lg, nt, wk, ft})
        }
    }
    
    
 } catch (error) {
    console.error(error.message)
    logger("SERVIDOR:postClientes").error(`Erro ao cadastrar o cliente ${error.message}`)

    if(error?.path){
      const rs = response("erro", 412, error.message);
      res.status(rs.statusCode).json(rs)        
    }else{  
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
    }
 } 
    
}

module.exports.postReferencia = async function(req, res, next) {
  try {

    let dados =  req.body

    
    let c1 = yup.string()
    if(dados?.opcional_1){
      yup.string().matches(/[a-zA-Z]{2,}=[\w{2,}\.\-\,\d\s]{1,}/ig, "Deve colocar correctamente dos dados (CAMPO 1). EX: campo=valor")
    }

    let c2 = yup.string()
    if(dados?.opcional_2){
      yup.string().matches(/[a-zA-Z]{2,}=[\w{2,}\.\-\,\d\s]{1,}/ig, "Deve colocar correctamente dos dados (CAMPO 2). EX: campo=valor")
    }

    let c3 = yup.string()
    if(dados?.opcional_3){
      yup.string().matches(/[a-zA-Z]{2,}=[\w{2,}\.\-\,\d\s]{1,}/ig, "Deve colocar correctamente dos dados (CAMPO 3). EX: campo=valor")
    }
    
    for (const prop in dados ) {    
      dados[prop] = dados[prop]
    }

    console.log(dados)

    if(!dados.hora_limite_pagamento) dados.hora_limite_pagamento =  "23:59"

    if(dados.montante_limitado_para_pagamento_longo && dados.montante_fixo) res.status(422).json({status:"erro", mensagem:"Montante fixo e o Montante limitado não podem ser paramentrizados juntos."})
    
    
    const refSchemaReg1 = yup.object().shape({
      entidade_cliente: yup.string().min(5).required(),
      tipo_de_registro: yup.number().min(1).max(3).required(),
      montante_fixo: yup.number().min(0).max(19999999),
      montante_limitado_para_pagamento_longo: yup.number().min(0),
      nib: yup.string().min(21),
      codigo_de_processamento: yup.string().required(),
      num_referencia: yup.string().min(9).max(15),
      indicador_de_produtos: yup.string().required(),
      data_limite_pagamento: yup.date().transform(parseDateString).required(),
      hora_limite_pagamento: yup.string().test(
        "date validation",
        "Invalid date or format. The string should be a valid HH:mm format.",
        val => {
          if (!val) {
            return true;
          }
          return moment(val, "HH:mm", true).isValid();
        }
      ),
      indicador_produto_id: yup.string().required(),
      gerado_por_afiliado: yup.string(),
      opcional_1: c1,
      opcional_2: c2,
      opcional_3: c3 
    })
    
    const refSchemaReg3 = yup.object().shape({
      entidade_cliente: yup.string().min(5).required(),
      tipo_de_registro:  yup.number().min(1).max(3).required(),
      codigo_de_processamento: yup.string().required(),
      num_referencia: yup.string().min(9).max(15),
      indicador_de_produtos: yup.string().required(),
      data_limite_pagamento: yup.date().transform(parseDateString).required(),
      data_inicio_de_pagamento: yup.date().transform(parseDateString).required(),
      montante_minimo: yup.number().min(100).required(),
      montante_maximo: yup.number().min(100).max(19999999).required(), 
      codigo_de_cliente: yup.string().required(),
      numero_de_linhas : yup.number().required().max(4),
      textos_para_talao: yup.string().required().max(40),
      indicador_produto_id: yup.number().required(),
      gerado_por_afiliado: yup.string(),
      opcional_1: c1,
      opcional_2: c2,
      opcional_3: c3
    
    })
    
    
    if(dados.tipo_de_registro == "1"){
      dados = {
            entidade_cliente: dados.entidade_cliente,
            tipo_de_registro: dados.tipo_de_registro,
            montante_fixo: dados.montante_fixo || 0,
            montante_limitado_para_pagamento_longo: dados.montante_limitado_para_pagamento_longo || 0,
            nib: dados.nib,
            codigo_de_processamento: dados.codigo_de_processamento,
            num_referencia: dados.num_referencia,
            indicador_de_produtos: dados.indicador_de_produtos,
            data_limite_pagamento: dados.data_limite_pagamento,
            hora_limite_pagamento: dados.hora_limite_pagamento,
            indicador_produto_id: dados.indicador_produto_id,
            gerado_por_afiliado: dados.gerado_por_afiliado || 0,
            opcional_1: dados.opcional_1,
            opcional_2: dados.opcional_2,
            opcional_3: dados.opcional_3,
        }
        
        const validar = await refSchemaReg1.validate(dados)
        console.log("CARREGAMENTO / PAGAMENTOS",validar)
        const result = await models.postReferencia({...validar, data_limite_pagamento: new Date(validar.data_limite_pagamento).toISOString().split('T')[0]}, req)  
        
        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        var ft = result.mft
        
        delete result.webhook
        delete result.logs
        delete result.notification
        delete result.mft
        
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){
            sendRequestOnMicroservices({lg, nt, wk, ft})
        }
                
    }
     
    if(dados.tipo_de_registro == "3"){
      dados = {
            entidade_cliente: dados.entidade_cliente,
            tipo_de_registro: dados.tipo_de_registro,
            codigo_de_processamento: dados.codigo_de_processamento,
            num_referencia: dados.num_referencia,
            indicador_de_produtos: dados.indicador_de_produtos,
            data_limite_pagamento: dados.data_limite_pagamento,
            data_inicio_de_pagamento: dados.data_inicio_de_pagamento,
            montante_minimo: dados.montante_minimo,
            montante_maximo: dados.montante_maximo,
            codigo_de_cliente: dados.codigo_de_cliente,
            numero_de_linhas : dados.numero_de_linhas,
            textos_para_talao: dados.textos_para_talao,
            indicador_produto_id: dados.indicador_produto_id,
            gerado_por_afiliado: dados.gerado_por_afiliado || 0,
            opcional_1: dados.opcional_1,
            opcional_2: dados.opcional_2,
            opcional_3: dados.opcional_3,
            
        }

        
        const validar = await refSchemaReg3.validate(dados)
        console.log("FACTURAS",validar)
        const result = await models.postReferencia({...validar, data_inicio_de_pagamento: new Date(validar.data_inicio_de_pagamento).toISOString().split('T')[0], data_limite_pagamento: new Date(validar.data_limite_pagamento).toISOString().split('T')[0]}, req)     
        
        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        var ft = result.mft
        
        delete result.webhook
        delete result.logs
        delete result.notification
        delete result.mft
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){
            sendRequestOnMicroservices({lg, nt, wk, ft})
        }
    }    
    
 } catch (error) {
    console.error(error.message)
    logger("SERVIDOR:postClientes").error(`Erro ao cadastrar o cliente ${error.message}`)

    if(error?.path){
      const rs = response("erro", 412, error.message);
      res.status(rs.statusCode).json(rs)        
    }else{  
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
    }
 } 
    
}

module.exports.patchReferencia = async function(req, res, next) {
  try {
    const {id_referencia} = req.params
    let dados = req.body

    let c1 = yup.string()
    if(dados?.opcional_1){
      yup.string().matches(/[a-zA-Z]{2,}=[\w{2,}\.\-\,\d\s]{1,}/ig, "Deve colocar correctamente dos dados (CAMPO 1). EX: campo=valor")
    }

    let c2 = yup.string()
    if(dados?.opcional_2){
      yup.string().matches(/[a-zA-Z]{2,}=[\w{2,}\.\-\,\d\s]{1,}/ig, "Deve colocar correctamente dos dados (CAMPO 2). EX: campo=valor")
    }

    let c3 = yup.string()
    if(dados?.opcional_3){
      yup.string().matches(/[a-zA-Z]{2,}=[\w{2,}\.\-\,\d\s]{1,}/ig, "Deve colocar correctamente dos dados (CAMPO 3). EX: campo=valor")
    }
    
    for (const prop in dados ) {    
      dados[prop] = dados[prop]
    }


    if(!dados.hora_limite_pagamento) dados.hora_limite_pagamento =  "23:59"

    if(dados.montante_limitado_para_pagamento_longo && dados.montante_fixo) res.status(422).json({status:"erro", mensagem:"Montante fixo e o Montante limitado não podem ser paramentrizados juntos."})
    
    
    const refSchemaReg1 = yup.object().shape({
      entidade_cliente: yup.string().min(5).required(),
      tipo_de_registro: yup.number().min(1).max(3).required(),
      montante_fixo: yup.number().min(0).max(19999999),
      montante_limitado_para_pagamento_longo: yup.number().min(0),
      nib: yup.string().min(21),
      codigo_de_processamento: yup.string().required(),
      num_referencia: yup.string().min(9).max(15),
      indicador_de_produtos: yup.string().required(),
      data_limite_pagamento: yup.date().transform(parseDateString).required(),
      hora_limite_pagamento: yup.string().test(
        "date validation",
        "Invalid date or format. The string should be a valid HH:mm format.",
        val => {
          if (!val) {
            return true;
          }
          return moment(val, "HH:mm", true).isValid();
        }
      ),
      indicador_produto_id: yup.string().required(),
      gerado_por_afiliado: yup.string(),
      opcional_1: c1,
      opcional_2: c2,
      opcional_3: c3 
    })
    
    const refSchemaReg3 = yup.object().shape({
      entidade_cliente: yup.string().min(5).required(),
      tipo_de_registro:  yup.number().min(1).max(3).required(),
      codigo_de_processamento: yup.string().required(),
      num_referencia: yup.string().min(9).max(15),
      indicador_de_produtos: yup.string().required(),
      data_limite_pagamento: yup.date().transform(parseDateString).required(),
      data_inicio_de_pagamento: yup.date().transform(parseDateString).required(),
      montante_minimo: yup.number().min(100).required(),
      montante_maximo: yup.number().min(100).max(19999999).required(), 
      codigo_de_cliente: yup.string().required(),
      numero_de_linhas : yup.number().required().max(4),
      textos_para_talao: yup.string().required().max(40),
      indicador_produto_id: yup.number().required(),
      gerado_por_afiliado: yup.string(),
      opcional_1: c1,
      opcional_2: c2,
      opcional_3: c3
    
    })
    
    
    if(dados.tipo_de_registro == "1"){
      dados = {
            entidade_cliente: dados.entidade_cliente,
            tipo_de_registro: dados.tipo_de_registro,
            montante_fixo: dados.montante_fixo || 0,
            montante_limitado_para_pagamento_longo: dados.montante_limitado_para_pagamento_longo || 0,
            nib: dados.nib,
            codigo_de_processamento: dados.codigo_de_processamento,
            num_referencia: dados.num_referencia,
            indicador_de_produtos: dados.indicador_de_produtos,
            data_limite_pagamento: dados.data_limite_pagamento,
            hora_limite_pagamento: dados.hora_limite_pagamento,
            indicador_produto_id: dados.indicador_produto_id,
            gerado_por_afiliado: dados.gerado_por_afiliado || 0,
            opcional_1: dados.opcional_1,
            opcional_2: dados.opcional_2,
            opcional_3: dados.opcional_3,
        }
        
        const validar = await refSchemaReg1.validate(dados)

        const result = await models.patchReferencia(id_referencia, {...validar, data_limite_pagamento: new Date(validar.data_limite_pagamento).toISOString().split('T')[0]}, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        var ft = result.mft
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){
            sendRequestOnMicroservices({lg, nt, wk, ft})
        }
                
    }
     
    if(dados.tipo_de_registro == "3"){
      dados = {
            entidade_cliente: dados.entidade_cliente,
            tipo_de_registro: dados.tipo_de_registro,
            codigo_de_processamento: dados.codigo_de_processamento,
            num_referencia: dados.num_referencia,
            indicador_de_produtos: dados.indicador_de_produtos,
            data_limite_pagamento: dados.data_limite_pagamento,
            data_inicio_de_pagamento: dados.data_inicio_de_pagamento,
            montante_minimo: dados.montante_minimo,
            montante_maximo: dados.montante_maximo,
            codigo_de_cliente: dados.codigo_de_cliente,
            numero_de_linhas : dados.numero_de_linhas,
            textos_para_talao: dados.textos_para_talao,
            indicador_produto_id: dados.indicador_produto_id,
            gerado_por_afiliado: dados.gerado_por_afiliado || 0,
            opcional_1: dados.opcional_1,
            opcional_2: dados.opcional_2,
            opcional_3: dados.opcional_3,
            
        }

        
        const validar = await refSchemaReg3.validate(dados)

        const result = await models.patchReferencia(id_referencia, validar, req)

        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        var ft = result.mft
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){
            sendRequestOnMicroservices({lg, nt, wk, ft})
        }

    }

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:postClientes").error(`Erro ao cadastrar o cliente ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
  }
    
}

module.exports.deleteReferencia = async function(req, res, next) {
  try {

      const {id_referencia} = req.params
      const result = await models.deleteReferencia(id_referencia, req)

      var wk = result.webhook
      var lg = result.logs
      var nt = result.notification
      var ft = result.mft
      
      delete result.webhook
      delete result.logs
      delete result.notification
      delete result.mft
      
      res.status(result.statusCode).json(result)
      if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk, ft})
      }

  } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:deleteClientes").error(`Erro ao deletar o cliente ${error.message}`)
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
  }
    
}