const models = require('../models/sms')
const response = require("../constants/response");
const yup = require('yup')
const axios = require('axios')
const logger = require('../services/loggerService'); 
const sendRequestOnMicroservices = require("../helpers/sendRequestOnMicroservices"); 


module.exports.smsRemententeEntidade = async function(req, res, next) {
  try{

      logger("SERVIDOR:smsRemententeEntidade").info(`Buscar dados remetente de sms`)
      const {entidade} = req.params   
    
      const updateStatus = await models.smsRemententeEntidade(entidade, req)
      res.status(updateStatus.statusCode).json(updateStatus)
      
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:smsRemententeEntidade").error(`Erro ao buscar remetente de sms ${error.message}`)
  }
    
}

module.exports.getClintesSMSGET = async function(req, res, next) {
  try{

      logger("SERVIDOR:ClintesSMSGET").info(`Buscar as mensagens de clientes`)
      const {pagina, limite, mensagem = ''} = req.query
      const results = await models.getClintesSMSGET(pagina, limite, mensagem, req)
      res.status(results.statusCode).json(results)
      
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:ClintesSMSGET").error(`Erro ao buscar as mensagens de clientes ${error.message}`)
  }
    
}

module.exports.getClintesSMSEntidadeGET = async function(req, res, next) {
  try{

      logger("SERVIDOR:ClintesSMSEntidadeGET").info(`Buscar as mensagens de sms do cliente`)
      const {entidade} = req.params    
      const {pagina, limite, mensagem = ''} = req.query
    
      const clintesSMSEntidade = await models.getClintesSMSEntidadeGET(entidade, pagina, limite, mensagem, req) 
      res.json(clintesSMSEntidade)
      
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:ClintesSMSEntidadeGET").error(`Erro ao buscar as mensagens de sms do cliente ${error.message}`)
  }
    
}

module.exports.getClintesSMS = async function(req, res, next) {
  try{
       
      logger("SERVIDOR:ClintesSMS").info(`Buscar dados configuração sms de cliente`)
      const {pagina, limite} = req.query

      const clientesSMS = await models.getClintesSMS(pagina, limite, req)
      res.status(clientesSMS.statusCode).json(clientesSMS)
      
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:ClintesSMS").error(`Erro ao buscar configuração sms de cliente ${error.message}`)
  }
    
}

module.exports.getClintesSMSEntidade = async function(req, res, next) {
  try{

      logger("SERVIDOR:ClintesSMSEntidade").info(`Buscar dados configuração sms de cliente`)
      const {entidade} = req.params 
      const {pagina, limite} = req.query   
    
      const smsEntidade = await models.getClintesSMSEntidade(entidade, pagina, limite, req)
      res.status(smsEntidade.statusCode).json(smsEntidade)
      
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:ClintesSMSEntidade").error(`Erro ao buscar configuração sms de cliente ${error.message}`)
  }
    
}

module.exports.smsRemententes = async function(req, res, next) {
  try{
    
    logger("SERVIDOR:smsRemententes").debug(`Busca HTTP do remetente`)
    const {data} = await axios.get(`${process.env.HOST_NOTIFICACAO}/remetentes`)
    const remetentes = []
    if(data){
      
      for (const it of data) {
        logger("SERVIDOR:smsRemententes").info(`Buscar dados remetentes de sms`)
        const status = await models.StatusRemetentes(it.id)
        
        if(status.mensagem.length) remetentes.push({...it, ...status.mensagem[0]})    
        else remetentes.push({...it})   
      }

      res.status(200).json(remetentes)
      return
    }

    const rs = response("erro", 204, ``);
    res.status(rs.statusCode).json(rs)
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:smsRemententes").error(`Erro ao buscar remetentes de sms ${error.message}`)
  }
    
}

module.exports.smsRemententeID = async function(req, res, next) {
  try{

    logger("SERVIDOR:smsRemententeID").info(`Buscar dados remetentes de sms`)
    const {id} = req.params
    
    logger("SERVIDOR:smsRemententeID").debug(`Busca HTTP do remetente`)
    const {data} = await axios.get(`${process.env.HOST_NOTIFICACAO}/remetente/${id}`)

    console.log(data)

    if(data?.id){
      const updateStatus = await models.updateStatusRemetente(id, data.status, req)
      res.status(updateStatus.statusCode).json(updateStatus)
      return
    }

    const rs = response("sucesso", 200, data);
    res.status(rs.statusCode).json(rs)
    
  } catch (error) {
    console.error(error.message)
    const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
    res.status(rs.statusCode).json(rs)
    logger("SERVIDOR:smsRemententeID").error(`Erro ao buscar remetente de sms ${error.message}`)
  }
    
}

module.exports.postClientesSMS = async function(req, res, next) {
  try{
    
    logger("SERVIDOR:ClientesSMS").info("Iniando cadastro template de sms para o cliente")
    const dados = req.body

    const schemaTemplateSMS = yup.object().shape({
      mensagem: yup.string().required(),
      env_entidade: yup.string().min(5).required(), 
      tipo_env: yup.mixed().oneOf(['Pagamentos','Referencias']).required(), 
    })

    logger("SERVIDOR:ClientesSMS").debug(`Á validar os dados ${JSON.stringify(dados)}`)
    const validar = await schemaTemplateSMS.validate(dados)

    const result = await models.postClientesSMS(validar.mensagem, validar.env_entidade, validar.tipo_env, req)
    
    var wk = result.webhook
    var lg = result.logs
    var nt = result.notification
    
    delete result.webhook
    delete result.logs
    delete result.notification
    
    res.status(result.statusCode).json(result)
    if(result.status == "sucesso"){
      sendRequestOnMicroservices({lg, nt, wk})
    }
    
  } catch (error) {
    console.error(error.message)
    logger("SERVIDOR:ClientesSMS").error(`Erro ao cadastrar template de sms para o cliente ${error.message}`)

    if(error?.path){
      const rs = response("erro", 412, error.message);
      res.status(rs.statusCode).json(rs)        
    }else{  
      const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
      res.status(rs.statusCode).json(rs)
    }
  }
    
}

module.exports.smsRementente = async function(req, res, next) {
    
   try {
   
      logger("SERVIDOR:smsRementente").info("Iniando cadastro template de sms para o cliente")
      const dados = req.body

      const schemaRemetenteSMS = yup.object().shape({
        nome_rementente: yup.string().max(11).required(),
        campo_envio: yup.mixed().oneOf(['opcional_1','opcional_2', 'opcional_3']).required(),
        entidade_remetente: yup.string().min(5).required()
      })

      logger("SERVIDOR:smsRementente").debug(`Á validar os dados ${JSON.stringify(dados)}`)
      const validar = await schemaRemetenteSMS.validate(dados)   

      logger("SERVIDOR:smsRementente").debug(`Inserção HTTP para criar remetente`)
      const rs = await axios.post(`${process.env.HOST_NOTIFICACAO}/criar-remetente`, {rementente: validar.nome_rementente})

      if(rs.data?.errors?.length){
        logger("SERVIDOR:smsRementente").warn(`Não se conseguir ter a requição HTTP com sucesso`)
        res.json({status: "erro", mensagem: rs.data.errors[0].message})
        return
      }

      if(rs.data.id){

        console.log(rs.data)
      
        const dadosRespose = {
          ...validar,
          id_api_rementente: rs.data.id,
          estado_remetente: rs.data.status
        }

        logger("SERVIDOR:smsRementente").info(`Inserção na base de dados para criar remetente`)
        const result = await models.smsRemetentes(dadosRespose, req) 
        
        var wk = result.webhook
        var lg = result.logs
        var nt = result.notification
        
        delete result.webhook
        delete result.logs
        delete result.notification
        
        res.status(result.statusCode).json(result)
        if(result.status == "sucesso"){
          sendRequestOnMicroservices({lg, nt, wk})
        }

      }else{
        const rs = response("erro", 409, `Não foi possivel criar seu rementente! Tente mais tarde`);
        res.status(rs.statusCode).json(rs)
        logger("SERVIDOR:smsRementente").warn(`Não foi possivel criar seu rementente! Tente mais tarde`)
      }
      
   } catch (error) {
      console.error(error.message)
      logger("SERVIDOR:smsRementente").error(`Erro ao criar remetente de sms ${error.message}`)

      if(error?.path){
        const rs = response("erro", 412, error.message);
        res.status(rs.statusCode).json(rs)        
      }else{  
        const rs = response("erro", 400, `Algo aconteceu. Tente de novo, ${error.message}`);
        res.status(rs.statusCode).json(rs)
      }
   }
    
}