const database = require('../config/database')
const bcrypt = require("bcryptjs");
const { v4: uuidv4 } = require('uuid');
const response = require("../constants/response");
const logger = require('../services/loggerService');
const pagination = require("../constants/pagination");
const paginationRecords = require("../helpers/paginationRecords");
const { afiliadosFilteres, afiliadosIdentificacaoFilteres } = require('../helpers/filterResponseSQL');

module.exports.getAfiliados = async function(pagina, limite, total_registros, nome_afiliado, email_afiliado, user_name, entidade) {
  try {

      logger("SERVIDOR:getAfiliados").debug("Buscar todos afiliados no banco de dados");
      const afiliados = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
      .whereLike("nome_afiliado",`%${nome_afiliado}%`)
      .whereLike("email_afiliado",`%${email_afiliado}%`)
      .whereLike("user_name",`%${user_name}%`)
      .whereLike("numero_entidade",`%${entidade}%`)
      .limit(total_registros || pagination.total_limite)
      .orderBy("id_afiliado", "DESC");

      const {registros} = paginationRecords(afiliados, pagina, limite)

      logger("SERVIDOR:getAfiliados").debug(`Buscar todos afiliados no banco de dados com limite de ${registros.limite} na pagina ${registros.count} de registros`);
      const afiliadosLimite = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
      .whereLike("nome_afiliado",`%${nome_afiliado}%`)
      .whereLike("email_afiliado",`%${email_afiliado}%`)
      .whereLike("user_name",`%${user_name}%`)
      .whereLike("numero_entidade",`%${entidade}%`)
      .limit(registros.limite)
      .offset(registros.count)
      .orderBy("id_afiliado", "DESC");

      const filtered = afiliadosFilteres(afiliadosLimite)

      registros.total_apresentados = afiliadosLimite.length
      registros.nome_afiliado = nome_afiliado
      registros.email_afiliado = email_afiliado
      registros.user_name = user_name
      registros.entidade = entidade

      logger("SERVIDOR:getAfiliados").info("Respondeu a requisição");
      const rs = response("sucesso", 200, filtered, 'json', {registros})
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:getAfiliados").error(`Erro ao buscar afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.getAfiliadosId = async function(id_afiliado) {
  try {

      logger("SERVIDOR:getAfiliadosId").debug("Buscar todos afiliados no banco de dados");
      const afiliados = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
      .where({ id_afiliado})
      .orderBy("id_afiliado", "DESC");

      const [filtered] = afiliadosFilteres(afiliados)

      logger("SERVIDOR:getAfiliadosId").info("Respondeu a requisição");
      const rs = response("sucesso", 200, filtered || {});
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:getAfiliados").error(`Erro ao buscar afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.getAfiliadosIdIdentificador = async function(id_identificacao_afiliados) {
  try {
      
      logger("SERVIDOR:getAfiliadosIdIdentificador").debug("Buscar todos afiliados no banco de dados");
      const afiliados = await database("clientes")
      .join("identificacao_afiliados", "identificacao_afiliados.entidade_das_afiliadas", "=", "clientes.numero_entidade")
      .where({ id_identificacao_afiliados })
      .orderBy("id_clientes", "DESC");

      const [filtered] = afiliadosIdentificacaoFilteres(afiliados)

      logger("SERVIDOR:getAfiliadosIdIdentificador").info("Respondeu a requisição");
      const rs = response("sucesso", 200, filtered || {});
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:getAfiliadosIdIdentificador").error(`Erro ao buscar afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.getAfiliadosIdComEntidade = async function(id_afiliado, entidade) {
  try {

      logger("SERVIDOR:getAfiliadosIdComEntidade").debug("Buscar todos afiliados no banco de dados");
      const afiliados = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
      .where({ id_afiliado})
      .andWhere({ numero_entidade: entidade })
      .orderBy("id_afiliado", "DESC");

      const [filtered] = afiliadosFilteres(afiliados)

      logger("SERVIDOR:getAfiliadosIdComEntidade").info("Respondeu a requisição");
      const rs = response("sucesso", 200, filtered || {});
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:getAfiliadosIdComEntidade").error(`Erro ao buscar afiliados pela entidade e o id do afiliado ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.getAfiliadosHash = async function(hash) {
  try {

      logger("SERVIDOR:getAfiliadosHash").debug("Verificar se existencia do afiliado")
      const resultAfiliado  = await database('afiliados')
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .where({endereco_mac_unico_afiliado:hash})
      .andWhere({activacao: 'true'});


      if(resultAfiliado.length == 0 ){
        logger("SERVIDOR:getAfiliadosHash").info("Afiliado bloqueado")
        const rs = response("erro", 403, "Afiliado bloqueado");
        return rs

      }else{

        logger("SERVIDOR:getAfiliadosHash").debug("Verificar se existencia do cliente")
        const [resultClientes]  = await database('clientes')
        .where({id_clientes: resultAfiliado[0].cliente_id})
        .andWhere({bloqueio: '1'});

        if(!resultClientes?.numero_entidade){
          logger("SERVIDOR:getAfiliadosHash").info("Cliente está bloqueado")
          const rs = response("erro", 403, "Cliente está bloqueado");
          return rs  
        }


        logger("SERVIDOR:getAfiliadosHash").debug("Buscar a configuração do afiliado")
        const [verifivarPagamentoTempoReal] = await database('configuracoes').where({cliente_entidade: resultClientes?.numero_entidade})    
        
        if((verifivarPagamentoTempoReal?.servico_pagamento_por_sector == "false") && (verifivarPagamentoTempoReal?.servico_gpo == "false")){
          logger("SERVIDOR:getAfiliadosHash").warn("Serviço de pagamento inactivo")
          const rs = response("erro", 423, "Serviço de pagamento inactivo"); 
          return rs     
        }

        if(verifivarPagamentoTempoReal?.tentativas_login > 0){

          logger("SERVIDOR:getAfiliadosHash").debug("Selecionar da base de dados")
          const [afiliados] = await database("afiliados")
          .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
          .join("configuracoes", "configuracoes.cliente_entidade", "=", "clientes.numero_entidade")
          .join("permissoes_afiliados", "afiliados.id_afiliado", "=", "permissoes_afiliados.afiliado")
          .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
          .where({endereco_mac_unico_afiliado:hash})
          .orderBy('id_afiliado','DESC')  
        
          logger("SERVIDOR:getAfiliadosHash").info("Respondeu a solicitação")
          const rs = response("sucesso", 200, afiliados, "json");          
          return rs

        }else{
          logger("SERVIDOR:getAfiliadosHash").warn("Cliente bloqueiado")
          const rs = response("erro", 403, "Seu perfil se encontra bloqueado. Por favor entre em contacto com administrador");
          return rs
        }

    }

} catch (erro) { 
    console.log(erro)
    logger("SERVIDOR:getAfiliadosHash").error(`Erro ao buscar clientes por hash ${erro.message}`)
    const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
    return rs
}
    
}

module.exports.getAfiliadosComEntidadeIdentificadores = async function(entidade, pagina, limite) {
  try {

      logger("SERVIDOR:getAfiliadosComEntidadeIdentificadores").debug("Buscar todos afiliados no banco de dados");
      const afiliados = await database("clientes")
      .join("identificacao_afiliados", "identificacao_afiliados.entidade_das_afiliadas", "=", "clientes.numero_entidade")
      .where({ numero_entidade: entidade })
      .orderBy("id_identificacao_afiliados", "DESC");

      const {registros} = paginationRecords(afiliados, pagina, limite)

      const afiliadosLimite = await database("clientes")
      .join("identificacao_afiliados", "identificacao_afiliados.entidade_das_afiliadas", "=", "clientes.numero_entidade")
      .where({ numero_entidade: entidade })
      .limit(registros.limite)
      .offset(registros.count)
      .orderBy("id_identificacao_afiliados", "DESC");

      const filtered = afiliadosIdentificacaoFilteres(afiliadosLimite)
      registros.total_apresentados = afiliadosLimite.length
      registros.entidade = entidade

      logger("SERVIDOR:getAfiliadosIdComEntidade").info("Respondeu a requisição");
      const rs = response("sucesso", 200, filtered, 'json', {registros});
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:getAfiliadosComEntidadeIdentificadores").error(`Erro ao buscar afiliados pela entidade com a identificação ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.getAfiliadosIdComEntidadeIdentificador = async function(id_identificacao_afiliados, entidade) {
  try {

      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificador").debug("Buscar todos afiliados no banco de dados");
      const afiliados = await database("clientes")
      .join("identificacao_afiliados", "identificacao_afiliados.entidade_das_afiliadas", "=", "clientes.numero_entidade")
      .where({ id_identificacao_afiliados })
      .andWhere({ numero_entidade: entidade })
      .orderBy("id_clientes", "DESC");

      const [filtered] = afiliadosIdentificacaoFilteres(afiliados)

      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificador").info("Respondeu a requisição");
      const rs = response("sucesso", 200, filtered || {});
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificador").error(`Erro ao buscar afiliados pela entidade com a identificação ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.getAfiliadosIdComEntidadePermissoes = async function(id_afiliado, entidade) {
  try {
    
      logger("SERVIDOR:getAfiliadosIdComEntidadePermissoes").debug("Buscar todos afiliados no banco de dados");
      const afiliados = await database("permissoes_afiliados")
      .join("afiliados", "permissoes_afiliados.afiliado", "=", "afiliados.id_afiliado")
      .join("clientes", "clientes.id_clientes", "=", "permissoes_afiliados.cliente")
      .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
      .where({ id_afiliado })
      .andWhere({ numero_entidade: entidade })
      .orderBy("id_afiliado", "DESC");

      const filtered = afiliadosFilteres(afiliados)

      logger("SERVIDOR:getAfiliadosIdComEntidadePermissoes").info("Respondeu a requisição");
      const rs = response("sucesso", 200, filtered);
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:getAfiliadosIdComEntidadePermissoes").error(`Erro ao buscar afiliados pela entidade com a e suas permissões ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.getAfiliadosIdComEntidadeIdentificadorAfiliados = async function(id_identificacao_afiliados, entidade, pagina, limite, nome_afiliado, email_afiliado, user_name) {
  try {
      
      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificadorAfiliados").debug("Buscar todos afiliados no banco de dados");
      const afiliados = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
      .whereLike("nome_afiliado",`%${nome_afiliado}%`)
      .whereLike("email_afiliado",`%${email_afiliado}%`)
      .whereLike("user_name",`%${user_name}%`)
      .where({ id_identificacao_afiliados })
      .andWhere({ numero_entidade: entidade })
      .orderBy("id_afiliado", "DESC");

      const {registros} = paginationRecords(afiliados, pagina, limite)

      const afiliadosLimite = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
      .whereLike("nome_afiliado",`%${nome_afiliado}%`)
      .whereLike("email_afiliado",`%${email_afiliado}%`)
      .whereLike("user_name",`%${user_name}%`)
      .where({ id_identificacao_afiliados })
      .andWhere({ numero_entidade: entidade })
      .limit(registros.limite)
      .offset(registros.count)
      .orderBy("id_afiliado", "DESC");

      const filtered = afiliadosFilteres(afiliadosLimite)
      registros.total_apresentados = afiliadosLimite.length
      registros.id_identificacao_afiliados = id_identificacao_afiliados
      registros.entidade = entidade
      registros.nome_afiliado = nome_afiliado
      registros.email_afiliado = email_afiliado
      registros.user_name = user_name

      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificadorAfiliados").info("Respondeu a requisição");
      const rs = response("sucesso", 200, filtered, 'json', {registros});
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:getAfiliadosIdComEntidadeIdentificadorAfiliados").error(`Erro ao buscar afiliados pela entidade ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.getAfiliadosEntidade = async function(entidade, pagina, limite, limite, nome_afiliado, email_afiliado, user_name) {
  try {

      logger("SERVIDOR:getAfiliadosEntidade").debug("Buscar todos afiliados no banco de dados");
      const afiliados = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
      .whereLike("nome_afiliado",`%${nome_afiliado}%`)
      .whereLike("email_afiliado",`%${email_afiliado}%`)
      .whereLike("user_name",`%${user_name}%`)
      .where({ numero_entidade: entidade })
      .orderBy("id_afiliado", "DESC");

      const {registros} = paginationRecords(afiliados, pagina, limite)

      const afiliadosLimite = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .join("identificacao_afiliados", "identificacao_afiliados.id_identificacao_afiliados", "=", "afiliados.grupo_identificacao")
      .whereLike("nome_afiliado",`%${nome_afiliado}%`)
      .whereLike("email_afiliado",`%${email_afiliado}%`)
      .whereLike("user_name",`%${user_name}%`)
      .where({ numero_entidade: entidade })
      .limit(registros.limite)
      .offset(registros.count)
      .orderBy("id_afiliado", "DESC");

      const filtered = afiliadosFilteres(afiliadosLimite)
      registros.total_apresentados = afiliadosLimite.length
      registros.entidade = entidade
      registros.nome_afiliado = nome_afiliado
      registros.email_afiliado = email_afiliado
      registros.user_name = user_name

      logger("SERVIDOR:getAfiliadosIdComEntidade").info("Respondeu a requisição");
      const rs = response("sucesso", 200, filtered, 'json', {registros});
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:getAfiliadosIdComEntidade").error(`Erro ao buscar afiliados pela entidade ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.postAfiliadosIdentificador = async function(dados, req) {
  try {

      logger("SERVIDOR:postAfiliadosIdentificador").debug("Buscar o identificador no banco de dados");
      const afiliados = await database("identificacao_afiliados").where({nome_identificacao: dados.nome_identificacao})
      .andWhere({entidade_das_afiliadas: dados.entidade_das_afiliadas})

      if (afiliados.length > 0){
        logger("SERVIDOR:postAfiliadosIdentificador").info("Identificação já cadastrado!")
        const rs = response("erro", 409, "Identificação já cadastrado!");
        return rs
      }

      logger("SERVIDOR:getAfiliadosEntidade").debug("Buscar o cliente no banco de dados");
      const cliente = await database("clientes").where({numero_entidade: dados.entidade_das_afiliadas});

      if (cliente.length < 1){
        logger("SERVIDOR:postAfiliadosIdentificador").info("Este cliente não existe!")
        const rs = response("erro", 406, "Este cliente não existe!");
        return rs
      }

      logger("SERVIDOR:postAfiliadosIdentificador").debug("Buscar todos afiliados no banco de dados");
      await database("identificacao_afiliados").insert(dados);

      logger("SERVIDOR:postAfiliadosIdentificador").info("Identificação de afiliados cadastrado com sucesso");
      const rs = response("sucesso", 201, "Identificação de afiliados cadastrado com sucesso", 'json', {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "IDENTIFICACAO_AFILIADOS", informacao: dados, entidade: cliente[0].numero_entidade}
      });
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:postAfiliadosIdentificador").error(`Erro ao cadastrar identificardor de afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.postAfiliados = async function(dados, req) {
  try {

      logger("SERVIDOR:postAfiliados").debug("Buscar os afiliados no banco de dados");
      const afiliados = await database("afiliados")
      .where({email_afiliado: dados.email_afiliado})
      .andWhere({user_name: dados.user_name})
      .andWhere({cliente_id: dados.cliente_id})

      if (afiliados.length > 0){
        logger("SERVIDOR:postAfiliados").info("Afiliado já cadastrado!")
        const rs = response("erro", 409, "Afiliado já cadastrado!");
        return rs
      }

      logger("SERVIDOR:postAfiliados").debug("Buscar o cliente no banco de dados");
      const afliadoGrupo = await database("identificacao_afiliados")
      .join("clientes", "clientes.numero_entidade", "=", "identificacao_afiliados.entidade_das_afiliadas")
      .where({id_identificacao_afiliados: dados.grupo_identificacao})
      .where({id_clientes: dados.cliente_id});

      if (afliadoGrupo.length < 1){
        logger("SERVIDOR:ClientesHash").info("Grupo de afliados inexistente!")
        const rs = response("erro", 409, "Grupo de afliados inexistente!");
        return rs
      }

      logger("SERVIDOR:postAfiliados").debug("Buscar o cliente no banco de dados");
      const cliente = await database("clientes")
      .where({id_clientes: dados.cliente_id});

      if (cliente.length < 1){
        logger("SERVIDOR:ClientesHash").info("Este cliente não existe!")
        const rs = response("erro", 406, "Este cliente não existe!");
        return rs
      }
        
      logger("SERVIDOR:postAfiliados").debug("Á cadastrar afiliados no banco de dados");
      await database("afiliados").insert(dados);

      logger("SERVIDOR:postAfiliados").info("Afiliado cadastrado com sucesso");
      const rs = response("sucesso", 201, "Afiliado cadastrado com sucesso", 'json', {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "AFILIADOS", informacao: dados, entidade: cliente[0].numero_entidade}
      });
      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:postAfiliados").error(`Erro ao cadastrar afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.postAfiliadosLogin = async function({ user_name, user_pass }, req) {
  try {
    
      logger("SERVIDOR:postAfiliadosLogin").debug("Buscar os afiliados no banco de dados");
      const result = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .join("configuracoes", "configuracoes.cliente_entidade", "=", "clientes.numero_entidade")
      .where({ user_name })
      .orWhere({ email_afiliado: user_name }); 
      

      if (result.length == 0) {
        logger("SERVIDOR:postAfiliadosLogin").info("Dados não encontrados")
        const rs = response("erro", 401, "Dados não encontrados");
        return rs
      } else {

          if (bcrypt.compareSync(user_pass, result[0].user_pass)) {


            if(result[0].activacao == "false"){
              logger("SERVIDOR:postAfiliadosLogin").info("Afiliado bloqueado")
              const rs = response("erro", 403, "Afiliado bloqueado");
              return rs
            }

            logger("SERVIDOR:postAfiliadosLogin").debug("Buscar a configuração do cliente")
            const [verifivarPagamentoTempoReal] = await database('configuracoes').where({cliente_entidade: result[0].numero_entidade})        

            if((verifivarPagamentoTempoReal?.servico_pagamento_por_sector == "false") && (verifivarPagamentoTempoReal?.servico_gpo == "false")){
              logger("SERVIDOR:postAfiliadosLogin").warn("Serviço de pagamento inactivo")
              const rs = response("erro", 423, "Serviço de pagamento inactivo");
              return rs     
            }

            const login = new Date().toISOString().split('.')[0].replace('T',' ')
            
            logger("SERVIDOR:postAfiliadosLogin").debug("Actualizar o hash de login")
            await database('afiliados')
            .where({user_name})
            .orWhere({email_afiliado: user_name})
            .update({ultimo_login_afiliado: login, endereco_mac_unico_afiliado: uuidv4()})

            logger("SERVIDOR:postAfiliadosLogin").debug("Buscar dados do afiliado")
            const [resultNew]  = await database('afiliados').select("endereco_mac_unico_afiliado AS hash", "numero_entidade AS entidade", "ultimo_login_afiliado AS login")
            .join("clientes","afiliados.cliente_id","=","clientes.id_clientes")
            .join("configuracoes","configuracoes.cliente_entidade","=","clientes.numero_entidade")
            .where({user_name})
            .orWhere({email_afiliado: user_name})

            logger("SERVIDOR:postAfiliadosLogin").info("Afiliado logado com sucesso");
            const rs = response("sucesso", 202,  {hash:resultNew.hash, ultimo_login: resultNew.login, entidade: resultNew.entidade, master: false}, 'json', {
              logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "LOGINUSER" , tabela: "AFILIADOS", informacao: {user_name}, entidade: result[0].numero_entidade}
            });

            return rs;

        } else{
          logger("SERVIDOR:postAfiliadosLogin").info("Autenticação incorrecta")
          const rs = response("erro", 403, "Autenticação incorrecta");
          return rs
        }
      }

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:postAfiliadosLogin").error(`Erro ao fazer login afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.postAfiliadosLogout = async function(id_afiliado, req) {
  try {

      logger("SERVIDOR:postAfiliadosLogout").debug("Buscar os afiliados no banco de dados");
      const result = await database("afiliados")
      .join("clientes", "clientes.id_clientes", "=", "afiliados.cliente_id")
      .where({id_afiliado});

      if (result.length == 0) {
          logger("SERVIDOR:postAfiliadosLogout").info("Afiliado não encontrado")
          const rs = response("erro", 409, "Afiliado não encontrado");
          return rs
      } else {

          logger("SERVIDOR:postAfiliadosLogout").info("Sessão terminada com sucesso");
          const rs = response("sucesso", 202,  "Sessão terminada com sucesso", 'json', {
            logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "LOGOUTUSER" , tabela: "AFILIADOS", informacao: {dados: result[0]}, entidade: result[0].numero_entidade}
          });

          return rs;
      }

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:postAfiliadosLogout").error(`Erro ao fazer logout afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.patchAfiliados = async function(id_afiliado, dados, req) {
  try {
   
      const actualizado_em_ = new Date()
      .toISOString()
      .replace("T", " ")
      .substr(0, 19);

      logger("SERVIDOR:patchAfiliados").debug("Buscar os afiliados no banco de dados");
      const afiliados = await database("afiliados")
      .where({ id_afiliado})

      if (afiliados.length == 0) {
        logger("SERVIDOR:patchAfiliados").info("Afiliado não encontrado")
        const rs = response("erro", 409, "Afiliado não encontrado");
        return rs
      } 
    
      logger("SERVIDOR:patchAfiliados").debug("Actualizar os afiliados no banco de dados");
      await database("afiliados")
      .where({ id_afiliado })
      .update({ ...dados, actualizado_em_ });

      logger("SERVIDOR:patchAfiliados").info("Actualização feita com sucesso");
      const rs = response("sucesso", 202,  "Actualização feita com sucesso", 'json', {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "AFILIADOS", informacao: {id_afiliado, ...dados}}
      });

      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:patchAfiliados").error(`Erro ao actualizar afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.patchAfiliadosPermissoes = async function(id_afiliado, dados, req) {
  try {
      
      const actualizado_em_p = new Date()
      .toISOString()
      .replace("T", " ")
      .substr(0, 19);

      logger("SERVIDOR:patchAfiliadosPermissoes").debug("Buscar os afiliados no banco de dados");
      const afiliados = await database("afiliados")
      .where({ id_afiliado})

      if (afiliados.length == 0) {
        logger("SERVIDOR:patchAfiliadosPermissoes").info("Afiliado não encontrado")
        const rs = response("erro", 409, "Afiliado não encontrado");
        return rs
      } 
    
      logger("SERVIDOR:patchAfiliadosPermissoes").debug("Actualizar os afiliados no banco de dados");
      await database("permissoes_afiliados")
      .where({ afiliado: id_afiliado })
      .update({ ...dados, actualizado_em_p });

      logger("SERVIDOR:patchAfiliadosPermissoes").info("Actualização feita com sucesso");
      const rs = response("sucesso", 202,  "Actualização feita com sucesso", 'json', {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "PERMISSOES_AFILIADOS", informacao: {id_afiliado, ...dados}}
      });

      return rs;

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:patchAfiliadosPermissoes").error(`Erro ao actualizar afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};

module.exports.deleteAfiliados = async function(id_afiliado, req) {
  try {

      logger("SERVIDOR:deleteAfiliados").debug("Buscar os afiliados no banco de dados");
      const afiliados = await database("afiliados")
      .where({ id_afiliado})

      if (afiliados.length == 0) {
        logger("SERVIDOR:deleteAfiliados").info("Afiliado não encontrado")
        const rs = response("erro", 409, "Afiliado não encontrado");
        return rs
      } 
    
      logger("SERVIDOR:deleteAfiliados").debug("Actualizar os afiliados no banco de dados");
      await database("afiliados")
      .where({ id_afiliado })
      .del();

      logger("SERVIDOR:deleteAfiliados").info("Usuario excluido feita com sucesso");
      const rs = response("sucesso", 202, "Usuario excluido feita com sucesso", 'json', {
        logs: {ip: req.ip, verbo_rota_API: req.method, rota_API: req.originalUrl, tipo: "DEFAULT" , tabela: "AFILIADOS", informacao: {id_afiliado, ...afiliados[0]}}
      });

      return rs;
      

  } catch (error) {
      console.log(error);
      logger("SERVIDOR:deleteAfiliados").error(`Erro ao deleter afiliados ${error.message}`)
      const rs = response("erro", 400, 'Algo aconteceu. Tente de novo');
      return rs
  }
};
